/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xerror.c
 * -----------------------------------------------------------------*/

#include <xtypedef.h>
#include <xdynamic.h>
#include <xerrcapi.h>
#include <xutil.h>

#include <xpattern.h> 

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

extern const char perfboard_err[];
extern const char *pcst_list[];
extern const char *verchk_list[];
extern const XErrorDescription CapiErrorStrings[];
extern const XErrorDescription FirmwareErrorStrings[];


#if 1
/* release case:
   bad messages are to be prefered over no messages */

#define BX_E_LAST_ENTRY    BX_E_COUNT
#define BX_EFW_LAST_ENTRY  BX_EFW_COUNT

#else

#define BX_E_LAST_ENTRY    BX_E_MAXERR
#define BX_EFW_LAST_ENTRY  BX_EFW_MAXERR

#endif


/* store an error in case we lost the handle */
static bx_lasterrtype es_handleless;
/* store an error in case we do have to make an close afterwards */
static bx_lasterrtype es_storeforclose;

/* Special string-formatting routine to handle 64/32 bit values */
static size_t errsprintf(char *buf, const char *format, bx_int64 par1, bx_int64 par2, 
                      bx_int64 par3, bx_int64 par4, bx_int64 par5) {
    char outformat[10];
    char *bufmerk=buf;
    char *tmpbuf;
    int parnr=0;
    int l,s,working;
    bx_int64 par[5];

    par[0]=par1;
    par[1]=par2;
    par[2]=par3;
    par[3]=par4;
    par[4]=par5;

    while (strchr(format,'%')) {
        memcpy(buf,format,strchr(format,'%')-format);
        buf+=strchr(format,'%')-format;
        format=strchr(format,'%')+1;
        strcpy(outformat,"%");
        working=TRUE;
        l=FALSE;
        s=FALSE;
        while (format[0] && working) {
            switch (format[0]) {
            case 'l': 
                l=TRUE;
                break;
            case 'h':
                s=TRUE;
                break;
            case 'c':
                tmpbuf=(char *)malloc(100*sizeof(char));
                strncat(outformat,&format[0],1);
                sprintf(tmpbuf,outformat,(int)par[parnr]);
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                working=FALSE;
                parnr++;
                break;
#ifndef UNIX
            case 'C':
                tmpbuf=(char *)malloc(100*sizeof(char));
                strncat(outformat,&format[0],1);
                sprintf(tmpbuf,outformat,(wint_t)par[parnr]);
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                parnr++;
                working=FALSE;
                break;
#endif
            case 'd':
            case 'i':
                tmpbuf=(char *)malloc(100*sizeof(char));
                strncat(outformat,&format[0],1);
                if (l) {
                    sprintf(tmpbuf,outformat,(long)par[parnr]);
                } else if (s) {
                    sprintf(tmpbuf,outformat,(short)par[parnr]);
                } else {
                    sprintf(tmpbuf,outformat,(int)par[parnr]);
                }
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                parnr++;
                working=FALSE;
                break;
            case 'o':
            case 'u':
            case 'x':
            case 'X':
                tmpbuf=(char *)malloc(100*sizeof(char));
                strncat(outformat,&format[0],1);
                if (l) {
                    sprintf(tmpbuf,outformat,(unsigned long)par[parnr]);
                } else if (s) {
                    sprintf(tmpbuf,outformat,(unsigned short)par[parnr]);
                } else {
                    sprintf(tmpbuf,outformat,(unsigned int)par[parnr]);
                }
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                parnr++;
                working=FALSE;
                break;
            case 'e':
            case 'E':
            case 'f':
            case 'g':
            case 'G':
                tmpbuf=(char *)malloc(100*sizeof(char));
                strncat(outformat,&format[0],1);
#ifdef UNIX
                sprintf(tmpbuf,outformat,(double)par[parnr]);
#else
                sprintf(tmpbuf,outformat,(double)((signed __int64)par[parnr]));
#endif
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                parnr++;
                working=FALSE;
                break;
            case 'n':
                parnr++;
                working=FALSE;
                break;
            case 'p':
                tmpbuf=(char *)malloc(100*sizeof(char));
                strncat(outformat,&format[0],1);
                sprintf(tmpbuf,outformat,(void *)par[parnr]);
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                parnr++;
                working=FALSE;
                break;
            case 's':
                tmpbuf=(char *)malloc((strlen((char *)par[parnr])+1)*sizeof(char));
                strncat(outformat,&format[0],1);
                sprintf(tmpbuf,outformat,(char *)par[parnr]);
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                working=FALSE;
                parnr++;
                break;
#ifndef UNIX
            case 'S':
                tmpbuf=(char *)malloc(wcslen((wchar_t *)par[parnr])*sizeof(char));
                strncat(outformat,&format[0],1);
                sprintf(tmpbuf,outformat,(wchar_t *)par[parnr]);
                strcpy(buf,tmpbuf);
                buf+=strlen(tmpbuf);
                free(tmpbuf);
                parnr++;
                working=FALSE;
                break;
#endif
            case '%':
                strcpy(buf,"%");
                buf++;
                working=FALSE;
                break;
            }
            strncat(outformat,&format[0],1);
            format++;
        }
    }
    strcat(buf,format);
    return (size_t)(buf-bufmerk)+strlen(buf);
}


static const XErrorDescription * ErrDescFromList(const bx_errtype errcode)
{
  int i;

  /* search through all capi errors */
  /* Rem: The last (maximum index) entry in CapiErrorStrings[] has the errorcode BX_E_COUNT (==BX_E_LAST_ENTRY) */ 
  i=-1;
  do
  {
    i++;
    if (CapiErrorStrings[i].errcode==errcode)
    {
      return &CapiErrorStrings[i];
    }
  } while (CapiErrorStrings[i].errcode!=BX_E_LAST_ENTRY);  /* must be != operator !! */
  
  
  /* ok, if not found, search through all firmware errors */
  /* Rem: The last (maximum index) entry in FirmwareErrorStrings[] has the errorcode BX_EFW_COUNT (==BX_EFW_LAST_ENTRY) */ 
  i=-1;
  do 
  {
    i++;
    if (FirmwareErrorStrings[i].errcode==errcode)
    {
      return &FirmwareErrorStrings[i];
    }
  } while (FirmwareErrorStrings[i].errcode != BX_EFW_LAST_ENTRY); /* must be != operator !! */
  
  /* We should never get here (granted by caller) !!! */

  /* CAUTION:
     To be correct, the following indexing is not allowed ! 
     But fortunately, index (39) and errorcode (BX_E_UNKNOWN_ERR) are (currently) identical in this case ! */
  return &CapiErrorStrings[BX_E_UNKNOWN_ERR]; 
}


/* --------------------------------------------------------------------------
 * TODO; this is a rather drastic way of recovering from a bad handle !!!
 * 
 * -------------------------------------------------------------------------- */

static void handle_problem(void)
{

#if 1
  bx_handletype handle;
  for (handle = 0; handle < BX_MAXHANDLES; ++handle)
  {
    if (bx_handlearray[handle].is_open)
    {
      (void) BestXLastErrorSet(handle, BX_E_BAD_HANDLE);
    }
  }

#else

  BESTX_FPRINTF (stderr, "Invalid handle used to set error.");
  exit (1);

#endif
  
}


/* --------------------------------------------------------------------------
 * Internally used error functions
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * This function is used by ALL parameter checking routines to record
 * the error if the  check fails.  It is aliased by the macro __FCT_PARAM_MSG
 * which is called by the BX_FCT_PARAM_CHK series of macros.
 * -------------------------------------------------------------------------- */

void EXPORT BestXFuncParamMsg(
    bx_handletype handle,
    bx_ccharptrtype pFuncName,
    bx_ccharptrtype param,
    bx_ccharptrtype reason)
{
  char s1[256];
  BESTX_STRCPY(s1, pFuncName);
  BESTX_STRCAT(s1, "@");
  BESTX_STRCAT(s1, param);
  BESTX_STRCAT(s1, "@");
  BESTX_STRCAT(s1, reason);
  BestXLastErrorParamStringSet(handle, s1);
}



/* --------------------------------------------------------------------------
 * Exported error functions
 * -------------------------------------------------------------------------- */

bx_ccharptrtype EXPORT BestXErrorStringGet(
    bx_errtype error
)
{
  /* NEW: Now with comments because of special request from Henrik, actually
   * I just changed the font color from white to black. CZ */
   /* Well Christian, sometimes a little comment helps a lot in understanding
   * of what was actually meant.... */

  bx_ccharptrtype msg;
  bx_handletype handle;
  /* scan through all handles to look for the error message */

  for (handle = 0; handle < BX_MAXHANDLES; ++handle)
  {
    if ((bx_handlearray[handle].is_open) &&
      (bx_handlearray[handle].lasterr.lasterr_err == error))
    {
      msg = bx_handlearray[handle].lasterr.lasterr_msg;
      goto msg_label;
    }
  }

  /* not found, maybe it's stored with a close ? */
  if (es_storeforclose.lasterr_err == error)
  {
    msg = es_storeforclose.lasterr_msg;
    goto msg_label;
  }

  /* not found, maybe it's a handleless problem */
  if (es_handleless.lasterr_err == error)
  {
    msg = es_handleless.lasterr_msg;
    goto msg_label;
  }

  /* moved before, TW + HL, not found, maybe it's stored with a close ? */
  /*if (es_storeforclose.lasterr_err == error)
  {
    msg = es_storeforclose.lasterr_msg;
    goto msg_label;
  } */


  /* no according error found, just return the format string */
  if (error > BX_E_FIRMWARE)
  {
    if (error > BX_EFW_LAST_ENTRY)
    {
      msg = ErrDescFromList(BX_E_UNKNOWN_ERR)->error;
    }
    else
    {
      msg = ErrDescFromList(error)->error;
    }
  }

  else
  {
    if (error > BX_E_LAST_ENTRY)
    {
      msg = ErrDescFromList(BX_E_UNKNOWN_ERR)->error;
    }
    else
    {
/*    if (error == BX_E_PERFBOARD_ERROR)
      {
        msg = perfboard_err;
      }
      else */
      {
        msg = ErrDescFromList(error)->error;
      }
    }
  }

msg_label:

  if (msg && !msg[0])
  {
    msg = NULL;
  }

  return msg;
}



void EXPORT BestXLastErrorParamSet(bx_handletype handle, bx_errparamtype index, bx_int32 errpar)
{
  assert(index < BX_ERROR_NUMPARAMS);

  if (__BX_HANDLECHECK)
  {
    bx_handlearray[handle].lasterr.lasterr_par[index] = errpar;
  }
  else
  {
    es_handleless.lasterr_par[index] = errpar;
    handle_problem();
  }
}


void EXPORT BestXLastErrorAllParamsSet(bx_handletype handle,
    bx_int64 errpar1, bx_int64 errpar2,
    bx_int64 errpar3, bx_int64 errpar4,
    bx_int64 errpar5)
{
  if (__BX_HANDLECHECK)
  {
    bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_1] = errpar1;
    bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_2] = errpar2;
    bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_3] = errpar3;
    bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_4] = errpar4;
    bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_5] = errpar5;
  }

  else
  {
    es_handleless.lasterr_par[BX_ERRPAR_1] = errpar1;
    es_handleless.lasterr_par[BX_ERRPAR_2] = errpar2;
    es_handleless.lasterr_par[BX_ERRPAR_3] = errpar3;
    es_handleless.lasterr_par[BX_ERRPAR_4] = errpar4;
    es_handleless.lasterr_par[BX_ERRPAR_5] = errpar5;
    handle_problem();
  }
}


void EXPORT BestXLastErrorAllParamsGet(bx_handletype handle,
    bx_int64ptr errpar1, bx_int64ptr errpar2,
    bx_int64ptr errpar3, bx_int64ptr errpar4,
    bx_int64ptr errpar5)
{
  if (__BX_HANDLECHECK)
  {
    *errpar1 = bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_1];
    *errpar2 = bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_2];
    *errpar3 = bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_3];
    *errpar4 = bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_4];
    *errpar5 = bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_5];
  }

  else
  {
    *errpar1 = es_handleless.lasterr_par[BX_ERRPAR_1];
    *errpar2 = es_handleless.lasterr_par[BX_ERRPAR_2];
    *errpar3 = es_handleless.lasterr_par[BX_ERRPAR_3];
    *errpar4 = es_handleless.lasterr_par[BX_ERRPAR_4];
    *errpar5 = es_handleless.lasterr_par[BX_ERRPAR_5];
    handle_problem();
  }
}


void EXPORT BestXLastErrorParamStringSet(bx_handletype handle, bx_ccharptrtype s)
{
  /* check before we BESTX_STRCPY into outer space */

  if (s==NULL || BESTX_STRLEN(s) > BX_LASTERR_STRING)
  {
    return;
  }

  if (__BX_HANDLECHECK)
  {
    BESTX_STRCPY(bx_handlearray[handle].lasterr.lasterr_str, s);
  }
  else
  {
    BESTX_STRCPY(es_handleless.lasterr_str, s);
  }
}


/* --------------------------------------------------------------------------
 * Set last error...will also set last error to BX_E_OK !!!
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXLastErrorSet(bx_handletype handle, bx_errtype err)
{
  int i, err_pos, buf_offset = 0;
  char *msg_buf;
  char str_buf[BX_LASTERR_STRING];
  char *str_ptr;
  char *str_ptr2;
  bx_errtype tmp_err;
  bx_lasterrtype *ref_lasterr;
  bx_lasterrtype buf_lasterr;
  const XErrorDescription * ed = NULL;

  ref_lasterr = ((__BX_HANDLECHECK) ?
    &bx_handlearray[handle].lasterr : &es_handleless);

  /* out of bounds ? */

  if ((err > BX_EFW_LAST_ENTRY) ||
    ((err <= BX_E_FIRMWARE) && (err > BX_E_LAST_ENTRY)))
  {
    BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32)err);
    (void) BestXLastErrorSet(handle, BX_E_UNKNOWN_ERR);
    ref_lasterr->lasterr_err = err;
    return err;
  }

  ed = ErrDescFromList(err); /*gets errname, error, errcode, lookup1 - lookup5 */

  if (ed && ed->error)
  {
    if (err)
    {
      /* add msg and setup msg_buf to point to the next empty character */
      buf_offset = BESTX_SPRINTF(ref_lasterr->lasterr_msg, "%s:\n", ed->errname);
      msg_buf = &ref_lasterr->lasterr_msg[buf_offset];
    }
    else
    {
      msg_buf = ref_lasterr->lasterr_msg;
      msg_buf[0] = '\0';
    }

    switch (err)
    {
      case BX_E_VALUE:
      case BX_E_RANGE:
      case BX_E_ALIGN:
      case BX_E_PARAM_NOT_EXIST:
      case BX_EFW_VALUE:
      case BX_EFW_RANGE:
      case BX_EFW_ALIGN:
      case BX_EFW_PARAM_NOT_EXIST:
        if (ref_lasterr->lasterr_par[BX_ERRPAR_1] == (bx_int32)BX_PARAM_CUSTOM)
        {
          if (bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_2] >
          (bx_int32)BX_PCST_LAST)
          {
            bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_2] =
             (bx_int32)BX_PCST_UNKNOWN;
          }
    
          errsprintf(msg_buf, ed->error,
            (bx_int64)pcst_list[(int) bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_2]],
            ref_lasterr->lasterr_par[BX_ERRPAR_3],
            ref_lasterr->lasterr_par[BX_ERRPAR_4],
            ref_lasterr->lasterr_par[BX_ERRPAR_5],
            0);
        }
        else
        {
          assert(ref_lasterr->lasterr_par[BX_ERRPAR_1] < (bx_int32)BX_PARAM_LAST);
          BESTX_MEMCPY(&buf_lasterr, ref_lasterr, sizeof buf_lasterr);
          (void) BestXDynamicErrorStringGet(
            handle,
            bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_1],
            bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_2],
            bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_3],
            bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_4],
            bx_handlearray[handle].lasterr.lasterr_par[BX_ERRPAR_5],
            err,
            &buf_lasterr.lasterr_msg[buf_offset]
            );
          BESTX_MEMCPY(ref_lasterr, &buf_lasterr, sizeof buf_lasterr);
        }

        break;


      case BX_E_PATT_SYNTAX:
      case BX_E_PATT_UNDEF_TOKEN:
      case BX_E_PATT_MTO_LISTSIGNAL:
        /* ATTENTION: recursive == don't touch!! */
        /* get additional information for a nice error message */
        /* TODO: How to make result of BestXPattGetEdit 'dont't touch'? */
     
        BESTX_MEMCPY(&buf_lasterr, ref_lasterr, sizeof buf_lasterr);
        tmp_err = BestXPattGet(handle,
          (bx_patttype) ref_lasterr->lasterr_par[BX_ERRPAR_1],
          (bx_charptrtype *) &str_ptr);
        BESTX_MEMCPY(ref_lasterr, &buf_lasterr, sizeof buf_lasterr);

        if (tmp_err)
        {
          /* everything changed, replace current error with the error from
           * BestXPattGetEdit */
          err = tmp_err;
        }

        else
        {
          BestXLastErrorParamStringSet(handle, str_ptr);

          /* following code is very similar to syntax error */
          msg_buf = ref_lasterr->lasterr_msg;
          errsprintf(str_buf, ed->error, ref_lasterr->lasterr_par[BX_ERRPAR_1],0,0,0,0);
          BESTX_STRCAT(msg_buf, str_buf);
          BESTX_STRCAT(msg_buf, "\n");
          BESTX_STRCAT(msg_buf, ref_lasterr->lasterr_str);
          BESTX_STRCAT(msg_buf, "\n");

          /* BX_ERRPAR_2 indicates the position of the Error in the Pattern string */

          err_pos = (int) ref_lasterr->lasterr_par[BX_ERRPAR_2];
          
          /* i starts at 2 on purpose because BX_ERRPAR_2 is off by 2 */

          for (i = 2; i < err_pos; ++i)
          {
            BESTX_STRCAT(msg_buf, " ");
          }

          BESTX_STRCAT(msg_buf, "^");
        }

        break;

      
      case BX_E_SYNTAX:
        msg_buf = ref_lasterr->lasterr_msg;
        BESTX_STRCAT(msg_buf, ed->error);
        BESTX_STRCAT(msg_buf, "\n");
        BESTX_STRCAT(msg_buf, ref_lasterr->lasterr_str);
        BESTX_STRCAT(msg_buf, "\n");

        for (i = 0; i < (int) ref_lasterr->lasterr_par[BX_ERRPAR_1]; ++i)
        {
          BESTX_STRCAT(msg_buf, " ");
        }

        BESTX_STRCAT(msg_buf, "^");
        break;

      case BX_E_FCT_PARAM:
        BESTX_STRCPY(str_buf, ref_lasterr->lasterr_str);
        str_ptr2 = str_ptr = strchr(str_buf, '@');

        if (str_ptr)
        {
          str_ptr2 = strchr(str_ptr, '@');
          *str_ptr++ = '\0';

          if (str_ptr2)
          {
            *str_ptr2++ = '\0';
          }
        }

        BESTX_SPRINTF(msg_buf, ed->error, str_buf,
          str_ptr ? str_ptr : "unknown",
          str_ptr2 ? str_ptr2 : "unknown reason");
        break;


      case BX_E_UNDERFLOW:
      case BX_E_OVERFLOW:
      case BX_E_FILE_OPEN:
      case BX_E_BAD_FILE_FORMAT:
        BESTX_SPRINTF(msg_buf, ed->error, ref_lasterr->lasterr_str);
        break;
      case BX_E_ERROR:
        errsprintf(msg_buf, ed->error,
          (bx_int64)ref_lasterr->lasterr_str,
          ref_lasterr->lasterr_par[BX_ERRPAR_1],0,0,0);
        break;
     
      case BX_E_DECODER:
      case BX_E_CONFIG_VALUE_INVALID:
      case BX_E_CONFIG_MASK_INVALID:
        {
        BESTX_SPRINTF(msg_buf, ed->error,
          ref_lasterr->lasterr_str);
        break;
        }
 
     case BX_E_TRANSF_CMD:
     case BX_E_INVALID_CASE_MSG:
     case BX_E_TBD_MSG:
     case BX_E_FILE_ITEM_NOT_FOUND:
        errsprintf(msg_buf, ed->error,
          ref_lasterr->lasterr_par[BX_ERRPAR_1],
          (bx_int64)ref_lasterr->lasterr_str,0,0,0);
        break;

      default: 
        errsprintf(msg_buf, ed->error,
          (ed->lookup1 ?
            (bx_int64) ed->lookup1[(int) bx_handlearray[handle].
              lasterr.lasterr_par[BX_ERRPAR_1]] :
            ref_lasterr->lasterr_par[BX_ERRPAR_1]),
          (ed->lookup2 ?
            (bx_int64) ed->lookup2[(int) bx_handlearray[handle].
              lasterr.lasterr_par[BX_ERRPAR_2]] :
            ref_lasterr->lasterr_par[BX_ERRPAR_2]),
          (ed->lookup3 ?
            (bx_int64) ed->lookup3[(int) bx_handlearray[handle].
              lasterr.lasterr_par[BX_ERRPAR_3]] :
            ref_lasterr->lasterr_par[BX_ERRPAR_3]),
          (ed->lookup4 ?
            (bx_int64) ed->lookup4[(int) bx_handlearray[handle].
              lasterr.lasterr_par[BX_ERRPAR_4]] :
            ref_lasterr->lasterr_par[BX_ERRPAR_4]),
          (ed->lookup5 ?
            (bx_int64) ed->lookup5[(int) bx_handlearray[handle].
              lasterr.lasterr_par[BX_ERRPAR_5]] :
            ref_lasterr->lasterr_par[BX_ERRPAR_5]));
        break;
    }                           /*lint !e788 not all enums used */
  }                             /* if (ed && ed->error) */
  else
  {
    ref_lasterr->lasterr_msg[0] = '\0';
  }

  ref_lasterr->lasterr_err = err;

  /* MUST return the same param we got in order for macros to work properly */
  return err;
}



bx_errtype EXPORT BestXLastErrorGet(bx_handletype handle)
{
  if (__BX_HANDLECHECK)
  {
    return bx_handlearray[handle].lasterr.lasterr_err;
  }

  return BX_E_BAD_HANDLE;
}


bx_ccharptrtype EXPORT BestXLastErrorStringGet(bx_handletype handle)
{
  bx_ccharptrtype msg;

  if (__BX_HANDLECHECK)
  {
    msg = bx_handlearray[handle].lasterr.lasterr_msg;
  }
  else
  {
    msg = BestXErrorStringGet(BX_E_BAD_HANDLE);
  }

  if (msg && !msg[0])
  {
    msg = "No errors found";
  }

  return msg;
}


void BestXLastErrorStore(bx_handletype handle, bx_errtype err)
{
  /* We're about to loose the handle but maybe still do need the last error
   * message. Put it in a single linked global list */

  if (!__BX_HANDLECHECK)
  {
    return;
  }

  if (bx_handlearray[handle].lasterr.lasterr_err == err)
  {
    /* seems okay, lets copy it into the handleless  */
    BESTX_MEMCPY(&es_storeforclose, &bx_handlearray[handle].lasterr,
      sizeof(bx_lasterrtype));
  }
}
